/*
-----------------------------------------------------------------------------
 FFT routines for C-Control Pro.
 Adapted from : Fixed-point FFT routines for megaAVRs, (C)ChaN, 2005
-----------------------------------------------------------------------------
 16bit fixed-point FFT performance with ATMega128 running at 14.756 MHz/
 internal SRAM.
 ADC sample rate = 8861.54 sps. Input decimation = 1 (no decimation).

  Points:   Input, Execute,  Output,    Total:  Throughput
   64pts:   7.4ms,   2.1ms    1.3ms,   10.8ms;     5.9kpps
  128pts:  14.7ms,   5.0ms,   2.7ms,   22.4ms:     5.7kpps
  256pts:  29.1ms,  11.1ms,   5.3ms,   45.5ms:     5.6kpps
-----------------------------------------------------------------------------

-------------------------------------------------------------------------------
 Set number of samples (FFT_N) below (64,128,256,512,1024).
 Conform this setting:
   - adapt the constant tables below
   - adapt FFT_N in FFT.asm
   - adapt the length of the arrays in the FFT function calls
-------------------------------------------------------------------------------
*/
#define FFT_N 256

void fft_input $asm("Tag_fft_input")
    (int p_bfly_buff[], int p_tbl_window[], byte decimate);

void fft_execute $asm("Tag_fft_execute")
    (int p_bfly_buff[], int p_tbl_cos_sin[]);

void fft_output $asm("Tag_fft_output")
    (int p_bfly_buff[], int p_tbl_bitrev[]);

flash int tbl_window[FFT_N] =
{
/*
Hamming window for FFT_N=1024:
     2621,  2621,  2622,  2623,  2625,  2628,  2631,  2635,  2639,  2644,  2649,
     2655,  2662,  2669,  2676,  2685,  2693,  2703,  2713,  2723,  2734,  2746,
     2758,  2771,  2784,  2798,  2812,  2827,  2843,  2859,  2876,  2893,  2910,
     2929,  2948,  2967,  2987,  3008,  3029,  3050,  3073,  3095,  3119,  3142,
     3167,  3192,  3217,  3243,  3270,  3297,  3325,  3353,  3382,  3411,  3441,
     3471,  3502,  3533,  3565,  3598,  3631,  3664,  3698,  3733,  3768,  3804,
     3840,  3877,  3914,  3952,  3990,  4029,  4068,  4108,  4148,  4189,  4230,
     4272,  4314,  4357,  4401,  4444,  4489,  4534,  4579,  4625,  4671,  4718,
     4765,  4813,  4861,  4910,  4959,  5009,  5059,  5110,  5161,  5213,  5265,
     5317,  5370,  5424,  5478,  5532,  5587,  5642,  5698,  5754,  5811,  5868,
     5926,  5984,  6042,  6101,  6160,  6220,  6280,  6341,  6402,  6464,  6525,
     6588,  6651,  6714,  6777,  6841,  6906,  6970,  7036,  7101,  7167,  7234,
     7300,  7368,  7435,  7503,  7571,  7640,  7709,  7779,  7849,  7919,  7989,
     8060,  8132,  8203,  8275,  8348,  8420,  8493,  8567,  8641,  8715,  8789,
     8864,  8939,  9015,  9090,  9166,  9243,  9320,  9397,  9474,  9552,  9630,
     9708,  9787,  9866,  9945, 10024, 10104, 10184, 10264, 10345, 10426, 10507,
    10588, 10670, 10752, 10834, 10917, 10999, 11082, 11166, 11249, 11333, 11417,
    11501, 11586, 11670, 11755, 11840, 11926, 12011, 12097, 12183, 12269, 12355,
    12442, 12529, 12616, 12703, 12790, 12878, 12966, 13054, 13142, 13230, 13318,
    13407, 13496, 13584, 13674, 13763, 13852, 13942, 14031, 14121, 14211, 14301,
    14391, 14481, 14572, 14662, 14753, 14844, 14935, 15026, 15117, 15208, 15299,
    15391, 15482, 15574, 15665, 15757, 15849, 15940, 16032, 16124, 16216, 16308,
    16400, 16493, 16585, 16677, 16769, 16862, 16954, 17046, 17139, 17231, 17324,
    17416, 17509, 17601, 17694, 17786, 17879, 17971, 18064, 18156, 18248, 18341,
    18433, 18526, 18618, 18710, 18802, 18895, 18987, 19079, 19171, 19263, 19355,
    19447, 19539, 19630, 19722, 19814, 19905, 19997, 20088, 20179, 20271, 20362,
    20453, 20543, 20634, 20725, 20815, 20906, 20996, 21086, 21176, 21266, 21356,
    21446, 21535, 21625, 21714, 21803, 21892, 21980, 22069, 22157, 22246, 22334,
    22422, 22509, 22597, 22684, 22772, 22859, 22945, 23032, 23118, 23204, 23290,
    23376, 23462, 23547, 23632, 23717, 23802, 23886, 23970, 24054, 24138, 24222,
    24305, 24388, 24471, 24553, 24635, 24717, 24799, 24880, 24962, 25042, 25123,
    25203, 25283, 25363, 25443, 25522, 25601, 25679, 25758, 25836, 25913, 25991,
    26068, 26144, 26221, 26297, 26373, 26448, 26523, 26598, 26673, 26747, 26820,
    26894, 26967, 27040, 27112, 27184, 27256, 27327, 27398, 27469, 27539, 27609,
    27678, 27747, 27816, 27884, 27952, 28020, 28087, 28154, 28220, 28286, 28352,
    28417, 28482, 28546, 28610, 28674, 28737, 28800, 28862, 28924, 28985, 29046,
    29107, 29167, 29227, 29286, 29345, 29404, 29462, 29519, 29576, 29633, 29689,
    29745, 29800, 29855, 29910, 29963, 30017, 30070, 30123, 30175, 30226, 30277,
    30328, 30378, 30428, 30477, 30526, 30574, 30622, 30669, 30716, 30762, 30808,
    30854, 30899, 30943, 30987, 31030, 31073, 31115, 31157, 31198, 31239, 31280,
    31319, 31359, 31397, 31436, 31473, 31511, 31547, 31583, 31619, 31654, 31689,
    31723, 31756, 31790, 31822, 31854, 31885, 31916, 31947, 31976, 32006, 32034,
    32063, 32090, 32117, 32144, 32170, 32196, 32220, 32245, 32269, 32292, 32315,
    32337, 32359, 32380, 32400, 32420, 32440, 32459, 32477, 32495, 32512, 32528,
    32545, 32560, 32575, 32590, 32603, 32617, 32629, 32642, 32653, 32664, 32675,
    32685, 32694, 32703, 32711, 32719, 32726, 32732, 32738, 32744, 32748, 32753,
    32756, 32759, 32762, 32764, 32765, 32766, 32766, 32766, 32765, 32764, 32762,
    32759, 32756, 32753, 32748, 32744, 32738, 32732, 32726, 32719, 32711, 32703,
    32694, 32685, 32675, 32664, 32653, 32642, 32629, 32617, 32603, 32590, 32575,
    32560, 32545, 32529, 32512, 32495, 32477, 32459, 32440, 32420, 32400, 32380,
    32359, 32337, 32315, 32292, 32269, 32245, 32221, 32196, 32170, 32144, 32117,
    32090, 32063, 32035, 32006, 31976, 31947, 31916, 31885, 31854, 31822, 31790,
    31757, 31723, 31689, 31654, 31619, 31584, 31547, 31511, 31473, 31436, 31397,
    31359, 31319, 31280, 31239, 31198, 31157, 31115, 31073, 31030, 30987, 30943,
    30899, 30854, 30808, 30763, 30716, 30669, 30622, 30574, 30526, 30477, 30428,
    30378, 30328, 30277, 30226, 30175, 30123, 30070, 30017, 29964, 29910, 29855,
    29800, 29745, 29689, 29633, 29576, 29519, 29462, 29404, 29345, 29286, 29227,
    29167, 29107, 29046, 28985, 28924, 28862, 28800, 28737, 28674, 28610, 28546,
    28482, 28417, 28352, 28286, 28220, 28154, 28087, 28020, 27952, 27884, 27816,
    27747, 27678, 27609, 27539, 27469, 27398, 27327, 27256, 27184, 27112, 27040,
    26967, 26894, 26820, 26747, 26673, 26598, 26523, 26448, 26373, 26297, 26221,
    26144, 26068, 25991, 25913, 25836, 25758, 25679, 25601, 25522, 25443, 25363,
    25283, 25203, 25123, 25042, 24962, 24880, 24799, 24717, 24635, 24553, 24471,
    24388, 24305, 24222, 24138, 24054, 23970, 23886, 23802, 23717, 23632, 23547,
    23462, 23376, 23291, 23205, 23118, 23032, 22945, 22859, 22772, 22684, 22597,
    22510, 22422, 22334, 22246, 22158, 22069, 21981, 21892, 21803, 21714, 21625,
    21535, 21446, 21356, 21266, 21176, 21086, 20996, 20906, 20815, 20725, 20634,
    20544, 20453, 20362, 20271, 20179, 20088, 19997, 19905, 19814, 19722, 19631,
    19539, 19447, 19355, 19263, 19171, 19079, 18987, 18895, 18803, 18710, 18618,
    18526, 18433, 18341, 18249, 18156, 18064, 17971, 17879, 17786, 17694, 17601,
    17509, 17416, 17324, 17231, 17139, 17047, 16954, 16862, 16769, 16677, 16585,
    16493, 16401, 16308, 16216, 16124, 16032, 15940, 15849, 15757, 15665, 15574,
    15482, 15391, 15299, 15208, 15117, 15026, 14935, 14844, 14753, 14663, 14572,
    14482, 14391, 14301, 14211, 14121, 14031, 13942, 13852, 13763, 13674, 13585,
    13496, 13407, 13318, 13230, 13142, 13054, 12966, 12878, 12790, 12703, 12616,
    12529, 12442, 12355, 12269, 12183, 12097, 12011, 11926, 11840, 11755, 11670,
    11586, 11501, 11417, 11333, 11249, 11166, 11083, 11000, 10917, 10834, 10752,
    10670, 10588, 10507, 10426, 10345, 10264, 10184, 10104, 10024,  9945,  9866,
     9787,  9708,  9630,  9552,  9474,  9397,  9320,  9243,  9167,  9090,  9015,
     8939,  8864,  8789,  8715,  8641,  8567,  8494,  8420,  8348,  8275,  8203,
     8132,  8060,  7989,  7919,  7849,  7779,  7709,  7640,  7571,  7503,  7435,
     7368,  7300,  7234,  7167,  7101,  7036,  6970,  6906,  6841,  6777,  6714,
     6651,  6588,  6526,  6464,  6402,  6341,  6280,  6220,  6160,  6101,  6042,
     5984,  5926,  5868,  5811,  5754,  5698,  5642,  5587,  5532,  5478,  5424,
     5370,  5317,  5265,  5213,	 5161,  5110,  5059,  5009,  4959,  4910,  4861,
     4813,  4765,  4718,  4671,  4625,  4579,  4534,  4489,  4445,  4401,  4357,
     4314,  4272,  4230,  4189,  4148,  4108,  4068,  4029,  3990,  3952,  3914,
     3877,  3840,  3804,  3768,  3733,  3699,  3664,  3631,  3598,  3565,  3533,
     3502,  3471,  3441,  3411,  3382,  3353,  3325,  3297,  3270,  3243,  3217,
     3192,  3167,  3142,  3119,  3095,  3073,  3050,  3029,  3008,  2987,  2967,
     2948,  2929,  2910,  2893,  2876,  2859,  2843,  2827,  2812,  2798,  2784,
     2771,  2758,  2746,  2734,  2723,  2713,  2703,  2693,  2685,  2676,  2669,
     2662,  2655,  2649,  2644,  2639,  2635,  2631,  2628,  2625,  2623,  2622,
     2621

Hamming window for FFT_N=512:
     2621,  2622,  2625,  2631,  2639,  2649,  2662,  2676,  2693,  2713,  2734,
     2758,  2784,  2812,  2843,  2876,  2910,  2948,  2987,  3029,  3073,  3119,
     3167,  3217,  3270,  3325,  3382,  3441,  3502,  3565,  3631,  3698,  3768,
     3840,  3914,  3990,  4068,  4148,  4230,  4314,  4401,  4489,  4579,  4671,
     4765,  4861,  4959,  5059,  5161,  5265,  5370,  5478,  5587,  5698,  5811,
     5926,  6042,  6160,  6280,  6402,  6525,  6651,  6777,  6906,  7036,  7167,
     7300,  7435,  7571,  7709,  7849,  7989,  8132,  8275,  8420,  8567,  8715,
     8864,  9015,  9166,  9320,  9474,  9630,  9787,  9945, 10104, 10264, 10426,
    10588, 10752, 10917, 11082, 11249, 11417, 11586, 11755, 11926, 12097, 12269,
    12442, 12616, 12790, 12966, 13142, 13318, 13496, 13674, 13852, 14031, 14211,
    14391, 14572, 14753, 14935, 15117, 15299, 15482, 15665, 15849, 16032, 16216,
    16400, 16585, 16769, 16954, 17139, 17324, 17509, 17694, 17879, 18064, 18248,
    18433, 18618, 18802, 18987, 19171, 19355, 19539, 19722, 19905, 20088, 20271,
    20453, 20634, 20815, 20996, 21176, 21356, 21535, 21714, 21892, 22069, 22246,
    22422, 22597, 22772, 22945, 23118, 23290, 23462, 23632, 23802, 23970, 24138,
    24305, 24471, 24635, 24799, 24962, 25123, 25283, 25443, 25601, 25758, 25913,
    26068, 26221, 26373, 26523, 26673, 26820, 26967, 27112, 27256, 27398, 27539,
    27678, 27816, 27952, 28087, 28220, 28352, 28482, 28610, 28737, 28862, 28985,
    29107, 29227, 29345, 29462, 29576, 29689, 29800, 29910, 30017, 30123, 30226,
    30328, 30428, 30526, 30622, 30716, 30808, 30899, 30987, 31073, 31157, 31239,
    31319, 31397, 31473, 31547, 31619, 31689, 31756, 31822, 31885, 31947, 32006,
    32063, 32117, 32170, 32220, 32269, 32315, 32359, 32400, 32440, 32477, 32512,
    32545, 32575, 32603, 32629, 32653, 32675, 32694, 32711, 32726, 32738, 32748,
    32756, 32762, 32765, 32766, 32765, 32762, 32756, 32748, 32738, 32726, 32711,
    32694, 32675, 32653, 32629, 32603, 32575, 32545, 32512, 32477, 32440, 32400,
    32359, 32315, 32269, 32221, 32170, 32117, 32063, 32006, 31947, 31885, 31822,
    31757, 31689, 31619, 31547, 31473, 31397, 31319, 31239, 31157, 31073, 30987,
    30899, 30808, 30716, 30622, 30526, 30428, 30328, 30226, 30123, 30017, 29910,
    29800, 29689, 29576, 29462, 29345, 29227, 29107, 28985, 28862, 28737, 28610,
    28482, 28352, 28220, 28087, 27952, 27816, 27678, 27539, 27398, 27256, 27112,
    26967, 26820, 26673, 26523, 26373, 26221, 26068, 25913, 25758, 25601, 25443,
    25283, 25123, 24962, 24799, 24635, 24471, 24305, 24138, 23970, 23802, 23632,
    23462, 23291, 23118, 22945, 22772, 22597, 22422, 22246, 22069, 21892, 21714,
    21535, 21356, 21176, 20996, 20815, 20634, 20453, 20271, 20088, 19905, 19722,
    19539, 19355, 19171, 18987, 18803, 18618, 18433, 18249, 18064, 17879, 17694,
    17509, 17324, 17139, 16954, 16769, 16585, 16401, 16216, 16032, 15849, 15665,
    15482, 15299, 15117, 14935, 14753, 14572, 14391, 14211, 14031, 13852, 13674,
    13496, 13318, 13142, 12966, 12790, 12616, 12442, 12269, 12097, 11926, 11755,
    11586, 11417, 11249, 11083, 10917, 10752, 10588, 10426, 10264, 10104,  9945,
     9787,  9630,  9474,  9320,  9167,  9015,  8864,  8715,  8567,  8420,  8275,
     8132,  7989,  7849,  7709,  7571,  7435,  7300,  7167,  7036,  6906,  6777,
     6651,  6526,  6402,  6280,  6160,  6042,  5926,  5811,  5698,  5587,  5478,
     5370,  5265,  5161,  5059,  4959,  4861,  4765,  4671,  4579,  4489,  4401,
     4314,  4230,  4148,  4068,  3990,  3914,  3840,  3768,  3699,  3631,  3565,
     3502,  3441,  3382,  3325,  3270,  3217,  3167,  3119,  3073,  3029,  2987,
     2948,  2910,  2876,  2843,  2812,  2784,  2758,  2734,  2713,  2693,  2676,
     2662,  2649,  2639,  2631,  2625,  2622

Hamming window for FFT_N=256:
*/
     2621,  2625,  2639,  2662,  2693,  2734,  2784,  2843,  2910,  2987,  3073,
     3167,  3270,  3382,  3502,  3631,  3768,  3914,  4068,  4230,  4401,  4579,
     4765,  4959,  5161,  5370,  5587,  5811,  6042,  6280,  6525,  6777,  7036,
     7300,  7571,  7849,  8132,  8420,  8715,  9015,  9320,  9630,  9945, 10264,
    10588, 10917, 11249, 11586, 11926, 12269, 12616, 12966, 13318, 13674, 14031,
    14391, 14753, 15117, 15482, 15849, 16216, 16585, 16954, 17324, 17694, 18064,
    18433, 18802, 19171, 19539, 19905, 20271, 20634, 20996, 21356, 21714, 22069,
    22422, 22772, 23118, 23462, 23802, 24138, 24471, 24799, 25123, 25443, 25758,
    26068, 26373, 26673, 26967, 27256, 27539, 27816, 28087, 28352, 28610, 28862,
    29107, 29345, 29576, 29800, 30017, 30226, 30428, 30622, 30808, 30987, 31157,
    31319, 31473, 31619, 31756, 31885, 32006, 32117, 32220, 32315, 32400, 32477,
    32545, 32603, 32653, 32694, 32726, 32748, 32762, 32766, 32762, 32748, 32726,
    32694, 32653, 32603, 32545, 32477, 32400, 32315, 32221, 32117, 32006, 31885,
    31757, 31619, 31473, 31319, 31157, 30987, 30808, 30622, 30428, 30226, 30017,
    29800, 29576, 29345, 29107, 28862, 28610, 28352, 28087, 27816, 27539, 27256,
    26967, 26673, 26373, 26068, 25758, 25443, 25123, 24799, 24471, 24138, 23802,
    23462, 23118, 22772, 22422, 22069, 21714, 21356, 20996, 20634, 20271, 19905,
    19539, 19171, 18803, 18433, 18064, 17694, 17324, 16954, 16585, 16216, 15849,
    15482, 15117, 14753, 14391, 14031, 13674, 13318, 12966, 12616, 12269, 11926,
    11586, 11249, 10917, 10588, 10264,  9945,  9630,  9320,  9015,  8715,  8420,
     8132,  7849,  7571,  7300,  7036,  6777,  6526,  6280,  6042,  5811,  5587,
     5370,  5161,  4959,  4765,  4579,  4401,  4230,  4068,  3914,  3768,  3631,
     3502,  3382,  3270,  3167,  3073,  2987,  2910,  2843,  2784,  2734,  2693,
     2662,  2639,  2625
/*

Hamming window for FFT_N=128
     2621,  2639,  2693,  2784,  2910,  3073,  3270,  3502,  3768,  4068,  4401,
     4765,  5161,  5587,  6042,  6525,  7036,  7571,  8132,  8715,  9320,  9945,
    10588, 11249, 11926, 12616, 13318, 14031, 14753, 15482, 16216, 16954, 17694,
    18433, 19171, 19905, 20634, 21356, 22069, 22772, 23462, 24138, 24799, 25443,
    26068, 26673, 27256, 27816, 28352, 28862, 29345, 29800, 30226, 30622, 30987,
    31319, 31619, 31885, 32117, 32315, 32477, 32603, 32694, 32748, 32766, 32748,
    32694, 32603, 32477, 32315, 32117, 31885, 31619, 31319, 30987, 30622, 30226,
    29800, 29345, 28862, 28352, 27816, 27256, 26673, 26068, 25443, 24799, 24138,
    23462, 22772, 22069, 21356, 20634, 19905, 19171, 18433, 17694, 16954, 16216,
    15482, 14753, 14031, 13318, 12616, 11926, 11249, 10588,  9945,  9320,  8715,
     8132,  7571,  7036,  6526,  6042,  5587,  5161,  4765,  4401,  4068,  3768,
     3502,  3270,  3073,  2910,  2784,  2693,  2639

Hamming window for FFT_N=64
     2621,  2693,  2910,  3270,  3768,  4401,  5161,  6042,  7036,  8132,  9320,
    10588, 11926, 13318, 14753, 16216, 17694, 19171, 20634, 22069, 23462, 24799,
    26068, 27256, 28352, 29345, 30226, 30987, 31619, 32117, 32477, 32694, 32766,
    32694, 32477, 32117, 31619, 30987, 30226, 29345, 28352, 27256, 26068, 24799,
    23462, 22069, 20634, 19171,	17694, 16216, 14753, 13318, 11926, 10588,  9320,
     8132,  7036,  6042,  5161,  4401,  3768,  3270,  2910,  2693
*/
};

// Table of cos(x),sin(x), 0 <= x < pi, in FFT_N/2 steps
flash int tbl_cos_sin[FFT_N] =
{
/*
Table of cos(x),sin(x) for FFT_N=1024
    32767,     0, 32766,   201, 32764,   402, 32761,   603, 32757,   804,
    32751,  1005, 32744,  1206, 32736,  1406, 32727,  1607, 32717,  1808,
    32705,  2009, 32692,  2209, 32678,  2410, 32662,  2610, 32646,  2811,
    32628,  3011, 32609,  3211, 32588,  3411, 32567,  3611, 32544,  3811,
    32520,  4011, 32495,  4210, 32468,  4409, 32441,  4608, 32412,  4807,
    32382,  5006, 32350,  5205, 32318,  5403, 32284,  5601, 32249,  5799,
    32213,  5997, 32176,  6195, 32137,  6392, 32097,  6589, 32056,  6786,
    32014,  6982, 31970,  7179, 31926,  7375, 31880,  7571, 31833,  7766,
    31785,  7961, 31735,  8156, 31684,  8351, 31633,  8545, 31580,  8739,
    31525,  8932, 31470,  9126, 31413,  9319, 31356,  9511, 31297,  9703,
    31236,  9895, 31175, 10087, 31113, 10278, 31049, 10469, 30984, 10659,
    30918, 10849, 30851, 11038, 30783, 11227, 30713, 11416, 30643, 11604,
    30571, 11792, 30498, 11980, 30424, 12166, 30349, 12353, 30272, 12539,
    30195, 12724, 30116, 12909, 30036, 13094, 29955, 13278, 29873, 13462,
    29790, 13645, 29706, 13827, 29621, 14009, 29534, 14191, 29446, 14372,
    29358, 14552, 29268, 14732, 29177, 14911, 29085, 15090, 28992, 15268,
    28897, 15446, 28802, 15623, 28706, 15799, 28608, 15975, 28510, 16150,
    28410, 16325, 28309, 16499, 28208, 16672, 28105, 16845, 28001, 17017,
    27896, 17189, 27790, 17360, 27683, 17530, 27575, 17699, 27466, 17868,
    27355, 18036, 27244, 18204, 27132, 18371, 27019, 18537, 26905, 18702,
    26789, 18867, 26673, 19031, 26556, 19194, 26437, 19357, 26318, 19519,
    26198, 19680, 26077, 19840, 25954, 20000, 25831, 20159, 25707, 20317,
    25582, 20474, 25456, 20631, 25329, 20787, 25201, 20942, 25072, 21096,
    24942, 21249, 24811, 21402, 24679, 21554, 24546, 21705, 24413, 21855,
    24278, 22004, 24143, 22153, 24006, 22301, 23869, 22448, 23731, 22594,
    23592, 22739, 23452, 22883, 23311, 23027, 23169, 23169, 23027, 23311,
    22883, 23452, 22739, 23592, 22594, 23731, 22448, 23869, 22301, 24006,
    22153, 24143, 22004, 24278, 21855, 24413, 21705, 24546, 21554, 24679,
    21402, 24811, 21249, 24942, 21096, 25072, 20942, 25201, 20787, 25329,
    20631, 25456, 20474, 25582, 20317, 25707, 20159, 25831, 20000, 25954,
    19840, 26077, 19680, 26198, 19519, 26318, 19357, 26437, 19194, 26556,
    19031, 26673, 18867, 26789, 18702, 26905, 18537, 27019, 18371, 27132,
    18204, 27244, 18036, 27355, 17868, 27466, 17699, 27575, 17530, 27683,
    17360, 27790, 17189, 27896, 17017, 28001, 16845, 28105, 16672, 28208,
    16499, 28309, 16325, 28410, 16150, 28510, 15975, 28608, 15799, 28706,
    15623, 28802, 15446, 28897, 15268, 28992, 15090, 29085, 14911, 29177,
    14732, 29268, 14552, 29358, 14372, 29446, 14191, 29534, 14009, 29621,
    13827, 29706, 13645, 29790, 13462, 29873, 13278, 29955, 13094, 30036,
    12909, 30116, 12724, 30195, 12539, 30272, 12353, 30349, 12166, 30424,
    11980, 30498, 11792, 30571, 11604, 30643, 11416, 30713, 11227, 30783,
    11038, 30851, 10849, 30918, 10659, 30984, 10469, 31049, 10278, 31113,
    10087, 31175,  9895, 31236,  9704, 31297,  9511, 31356,  9319, 31413,
     9126, 31470,  8933, 31525,  8739, 31580,  8545, 31633,  8351, 31684,
     8156, 31735,  7961, 31785,  7766, 31833,  7571, 31880,  7375, 31926,
     7179, 31970,  6983, 32014,  6786, 32056,  6589, 32097,  6392, 32137,
     6195, 32176,  5997, 32213,  5799, 32249,  5601, 32284,  5403, 32318,
     5205, 32350,  5006, 32382,  4807, 32412,  4608, 32441,  4409, 32468,
     4210, 32495,  4011, 32520,  3811, 32544,  3611, 32567,  3411, 32588,
     3211, 32609,  3011, 32628,  2811, 32646,  2610, 32662,  2410, 32678,
     2209, 32692,  2009, 32705,  1808, 32717,  1607, 32727,  1407, 32736,
     1206, 32744,  1005, 32751,   804, 32757,   603, 32761,   402, 32764,
      201, 32766,     0, 32766,  -201, 32766,  -402, 32764,  -603, 32761,
     -804, 32757, -1005, 32751, -1206, 32744, -1406, 32736, -1607, 32727,
    -1808, 32717, -2009, 32705, -2209, 32692, -2410, 32678, -2610, 32662,
    -2811, 32646, -3011, 32628, -3211, 32609, -3411, 32588, -3611, 32567,
    -3811, 32544, -4010, 32520, -4210, 32495, -4409, 32468, -4608, 32441,
    -4807, 32412, -5006, 32382, -5205, 32350, -5403, 32318, -5601, 32284,
    -5799, 32249, -5997, 32213, -6195, 32176, -6392, 32137, -6589, 32097,
    -6786, 32056, -6982, 32014, -7179, 31970, -7375, 31926, -7571, 31880,
    -7766, 31833, -7961, 31785, -8156, 31735, -8351, 31684, -8545, 31633,
    -8739, 31580, -8932, 31525, -9126, 31470, -9319, 31413, -9511, 31356,
    -9703, 31297, -9895, 31236,-10087, 31175,-10278, 31113,-10469, 31049,
   -10659, 30984,-10849, 30918,-11038, 30851,-11227, 30783,-11416, 30713,
   -11604, 30643,-11792, 30571,-11979, 30498,-12166, 30424,-12353, 30349,
   -12539, 30272,-12724, 30195,-12909, 30116,-13094, 30036,-13278, 29955,
   -13462, 29873,-13645, 29790,-13827, 29706,-14009, 29621,-14191, 29534,
   -14372, 29446,-14552, 29358,-14732, 29268,-14911, 29177,-15090, 29085,
   -15268, 28992,-15446, 28897,-15623, 28802,-15799, 28706,-15975, 28608,
   -16150, 28510,-16325, 28410,-16499, 28309,-16672, 28208,-16845, 28105,
   -17017, 28001,-17189, 27896,-17360, 27790,-17530, 27683,-17699, 27575,
   -17868, 27466,-18036, 27355,-18204, 27244,-18371, 27132,-18537, 27019,
   -18702, 26905,-18867, 26789,-19031, 26673,-19194, 26556,-19357, 26438,
   -19519, 26318,-19680, 26198,-19840, 26077,-20000, 25954,-20159, 25831,
   -20317, 25707,-20474, 25582,-20631, 25456,-20787, 25329,-20942, 25201,
   -21096, 25072,-21249, 24942,-21402, 24811,-21554, 24679,-21705, 24546,
   -21855, 24413,-22004, 24278,-22153, 24143,-22301, 24006,-22448, 23869,
   -22594, 23731,-22739, 23592,-22883, 23452,-23027, 23311,-23169, 23169,
   -23311, 23027,-23452, 22883,-23592, 22739,-23731, 22594,-23869, 22448,
   -24006, 22301,-24143, 22153,-24278, 22005,-24413, 21855,-24546, 21705,
   -24679, 21554,-24811, 21402,-24942, 21249,-25072, 21096,-25201, 20942,
   -25329, 20787,-25456, 20631,-25582, 20474,-25707, 20317,-25831, 20159,
   -25954, 20000,-26077, 19840,-26198, 19680,-26318, 19519,-26437, 19357,
   -26556, 19194,-26673, 19031,-26789, 18867,-26905, 18702,-27019, 18537,
   -27132, 18371,-27244, 18204,-27355, 18036,-27466, 17868,-27575, 17699,
   -27683, 17530,-27790, 17360,-27896, 17189,-28001, 17017,-28105, 16845,
   -28207, 16672,-28309, 16499,-28410, 16325,-28510, 16150,-28608, 15975,
   -28706, 15799,-28802, 15623,-28897, 15446,-28992, 15268,-29085, 15090,
   -29177, 14911,-29268, 14732,-29358, 14552,-29446, 14372,-29534, 14191,
   -29620, 14009,-29706, 13827,-29790, 13645,-29873, 13462,-29955, 13278,
   -30036, 13094,-30116, 12910,-30195, 12724,-30272, 12539,-30349, 12353,
   -30424, 12167,-30498, 11980,-30571, 11792,-30643, 11604,-30713, 11416,
   -30783, 11228,-30851, 11038,-30918, 10849,-30984, 10659,-31049, 10469,
   -31113, 10278,-31175, 10087,-31236,  9895,-31297,  9704,-31356,  9511,
   -31413,  9319,-31470,  9126,-31525,  8933,-31580,  8739,-31633,  8545,
   -31684,  8351,-31735,  8156,-31784,  7961,-31833,  7766,-31880,  7571,
   -31926,  7375,-31970,  7179,-32014,  6983,-32056,  6786,-32097,  6589,
   -32137,  6392,-32175,  6195,-32213,  5997,-32249,  5799,-32284,  5601,
   -32318,  5403,-32350,  5205,-32382,  5006,-32412,  4807,-32441,  4609,
   -32468,  4409,-32495,  4210,-32520,  4011,-32544,  3811,-32567,  3611,
   -32588,  3411,-32609,  3211,-32628,  3011,-32646,  2811,-32662,  2611,
   -32678,  2410,-32692,  2210,-32705,  2009,-32717,  1808,-32727,  1607,
   -32736,  1407,-32744,  1206,-32751,  1005,-32757,   804,-32761,   603,
   -32764,   402,-32766,   201

Table of cos(x),sin(x) for FFT_N=512
    32767,     0, 32764,   402, 32757,   804, 32744,  1206, 32727,  1607,
    32705,  2009, 32678,  2410, 32646,  2811, 32609,  3211, 32567,  3611,
    32520,  4011, 32468,  4409, 32412,  4807, 32350,  5205, 32284,  5601,
    32213,  5997, 32137,  6392, 32056,  6786, 31970,  7179, 31880,  7571,
    31785,  7961, 31684,  8351, 31580,  8739, 31470,  9126, 31356,  9511,
    31236,  9895, 31113, 10278, 30984, 10659, 30851, 11038, 30713, 11416,
    30571, 11792, 30424, 12166,	30272, 12539, 30116, 12909, 29955, 13278,
    29790, 13645, 29621, 14009, 29446, 14372, 29268, 14732, 29085, 15090,
    28897, 15446, 28706, 15799, 28510, 16150, 28309, 16499, 28105, 16845,
    27896, 17189, 27683, 17530, 27466, 17868, 27244, 18204, 27019, 18537,
    26789, 18867, 26556, 19194, 26318, 19519, 26077, 19840, 25831, 20159,
    25582, 20474, 25329, 20787, 25072, 21096, 24811, 21402, 24546, 21705,
    24278, 22004, 24006, 22301, 23731, 22594, 23452, 22883, 23169, 23169,
    22883, 23452, 22594, 23731, 22301, 24006, 22004, 24278, 21705, 24546,
    21402, 24811, 21096, 25072,	20787, 25329, 20474, 25582, 20159, 25831,
    19840, 26077, 19519, 26318, 19194, 26556, 18867, 26789, 18537, 27019,
    18204, 27244, 17868, 27466, 17530, 27683, 17189, 27896, 16845, 28105,
    16499, 28309, 16150, 28510, 15799, 28706, 15446, 28897, 15090, 29085,
    14732, 29268, 14372, 29446, 14009, 29621, 13645, 29790, 13278, 29955,
    12909, 30116, 12539, 30272, 12166, 30424, 11792, 30571, 11416, 30713,
    11038, 30851, 10659, 30984, 10278, 31113,  9895, 31236,  9511, 31356,
     9126, 31470,  8739, 31580,  8351, 31684,  7961, 31785,  7571, 31880,
     7179, 31970,  6786, 32056,  6392, 32137,  5997, 32213,  5601, 32284,
     5205, 32350,  4807, 32412,  4409, 32468,  4011, 32520,  3611, 32567,
     3211, 32609,  2811, 32646,  2410, 32678,  2009, 32705,  1607, 32727,
     1206, 32744,   804, 32757,   402, 32764,     0, 32766,  -402, 32764,
     -804, 32757, -1206, 32744, -1607, 32727, -2009, 32705, -2410, 32678,
    -2811, 32646, -3211, 32609, -3611, 32567, -4010, 32520, -4409, 32468,
    -4807, 32412, -5205, 32350, -5601, 32284, -5997, 32213, -6392, 32137,
    -6786, 32056, -7179, 31970, -7571, 31880, -7961, 31785, -8351, 31684,
    -8739, 31580, -9126, 31470, -9511, 31356, -9895, 31236,-10278, 31113,
   -10659, 30984,-11038, 30851,-11416, 30713,-11792, 30571,-12166, 30424,
   -12539, 30272,-12909, 30116,-13278, 29955,-13645, 29790,-14009, 29621,
   -14372, 29446,-14732, 29268,-15090, 29085,-15446, 28897,-15799, 28706,
   -16150, 28510,-16499, 28309,-16845, 28105,-17189, 27896,-17530, 27683,
   -17868, 27466,-18204, 27244,-18537, 27019,-18867, 26789,-19194, 26556,
   -19519, 26318 -19840, 26077,-20159, 25831,-20474, 25582,-20787, 25329,
   -21096, 25072,-21402, 24811,-21705, 24546,-22004, 24278,-22301, 24006,
   -22594, 23731,-22883, 23452,-23169, 23169,-23452, 22883,-23731, 22594,
   -24006, 22301,-24278, 22005,-24546, 21705,-24811, 21402,-25072, 21096,
   -25329, 20787,-25582, 20474,-25831, 20159,-26077, 19840,-26318, 19519,
   -26556, 19194,-26789, 18867,-27019, 18537,-27244, 18204,-27466, 17868,
   -27683, 17530,-27896, 17189,-28105, 16845,-28309, 16499,-28510, 16150,
   -28706, 15799,-28897, 15446,-29085, 15090,-29268, 14732,-29446, 14372,
   -29620, 14009,-29790, 13645,-29955, 13278,-30116, 12910,-30272, 12539,
   -30424, 12167,-30571, 11792,-30713, 11416,-30851, 11038,-30984, 10659,
   -31113, 10278,-31236,  9895,-31356,  9511,-31470,  9126,-31580,  8739,
   -31684,  8351,-31784,  7961,-31880,  7571,-31970,  7179,-32056,  6786,
   -32137,  6392,-32213,  5997,-32284,  5601,-32350,  5205,-32412,  4807,
   -32468,  4409,-32520,  4011,-32567,  3611,-32609,  3211,-32646,  2811,
   -32678,  2410,-32705,  2009,-32727,  1607,-32744,  1206,-32757,   804,
   -32764,   402

Table of cos(x),sin(x) for FFT_N=256
*/
    32767,     0, 32757,   804, 32727,  1607, 32678,  2410, 32609,  3211,
    32520,  4011, 32412,  4807, 32284,  5601, 32137,  6392, 31970,  7179,
    31785,  7961, 31580,  8739, 31356,  9511, 31113, 10278, 30851, 11038,
    30571, 11792, 30272, 12539, 29955, 13278, 29621, 14009, 29268, 14732,
    28897, 15446, 28510, 16150, 28105, 16845, 27683, 17530, 27244, 18204,
    26789, 18867, 26318, 19519, 25831, 20159, 25329, 20787, 24811, 21402,
    24278, 22004, 23731, 22594,	23169, 23169, 22594, 23731, 22004, 24278,
    21402, 24811, 20787, 25329, 20159, 25831, 19519, 26318, 18867, 26789,
    18204, 27244, 17530, 27683, 16845, 28105, 16150, 28510, 15446, 28897,
    14732, 29268, 14009, 29621, 13278, 29955, 12539, 30272, 11792, 30571,
    11038, 30851, 10278, 31113,  9511, 31356,  8739, 31580,  7961, 31785,
     7179, 31970,  6392, 32137,  5601, 32284,  4807, 32412,  4011, 32520,
     3211, 32609,  2410, 32678,  1607, 32727,   804, 32757,     0, 32766,
     -804, 32757, -1607, 32727, -2410, 32678, -3211, 32609, -4010, 32520,
    -4807, 32412, -5601, 32284,	-6392, 32137, -7179, 31970, -7961, 31785,
    -8739, 31580, -9511, 31356,-10278, 31113,-11038, 30851,-11792, 30571,
   -12539, 30272,-13278, 29955,-14009, 29621,-14732, 29268,-15446, 28897,
   -16150, 28510,-16845, 28105,-17530, 27683,-18204, 27244,-18867, 26789,
   -19519, 26318,-20159, 25831,-20787, 25329,-21402, 24811,-22004, 24278,
   -22594, 23731,-23169, 23169,-23731, 22594,-24278, 22005,-24811, 21402,
   -25329, 20787,-25831, 20159,-26318, 19519,-26789, 18867,-27244, 18204,
   -27683, 17530,-28105, 16845,-28510, 16150,-28897, 15446,-29268, 14732,
   -29620, 14009,-29955, 13278,-30272, 12539,-30571, 11792,-30851, 11038,
   -31113, 10278,-31356,  9511,-31580,  8739,-31784,  7961,-31970,  7179,
   -32137,  6392,-32284,  5601,-32412,  4807,-32520,  4011,-32609,  3211,
   -32678,  2410,-32727,  1607,-32757,   804
/*

Table of cos(x),sin(x) for FFT_N=128
    32767,     0, 32727,  1607, 32609,  3211, 32412,  4807, 32137,  6392,
    31785,  7961, 31356,  9511, 30851, 11038, 30272, 12539, 29621, 14009,
    28897, 15446, 28105, 16845, 27244, 18204, 26318, 19519, 25329, 20787,
    24278, 22004, 23169, 23169, 22004, 24278, 20787, 25329, 19519, 26318,
    18204, 27244, 16845, 28105, 15446, 28897, 14009, 29621, 12539, 30272,
    11038, 30851,  9511, 31356,  7961, 31785,  6392, 32137,  4807, 32412,
     3211, 32609,  1607, 32727,	    0, 32766, -1607, 32727, -3211, 32609,
    -4807, 32412, -6392, 32137, -7961, 31785, -9511, 31356,-11038, 30851,
   -12539, 30272,-14009, 29621,-15446, 28897,-16845, 28105,-18204, 27244,
   -19519, 26318,-20787, 25329,-22004, 24278,-23169, 23169,-24278, 22005,
   -25329, 20787,-26318, 19519,-27244, 18204,-28105, 16845,-28897, 15446,
   -29620, 14009,-30272, 12539,-30851, 11038,-31356,  9511,-31784,  7961,
   -32137,  6392, -32412, 4807,-32609,  3211,-32727,  1607

Table of cos(x),sin(x) for FFT_N=64
    32767,     0, 32609,  3211, 32137,  6392, 31356,  9511, 30272, 12539,
    28897, 15446, 27244, 18204, 25329, 20787, 23169, 23169, 20787, 25329,
    18204, 27244, 15446, 28897, 12539, 30272,  9511, 31356,  6392, 32137,
     3211, 32609,     0, 32766, -3211, 32609, -6392, 32137, -9511, 31356,
   -12539, 30272,-15446, 28897,-18204, 27244,-20787, 25329,-23169, 23169,
   -25329, 20787,-27244, 18204,-28897, 15446,-30272, 12539,-31356,  9511,
   -32137,  6392,-32609,  3211
*/
};

flash int tbl_bitrev[FFT_N/2] =
{
/*
Table for bit reversal for FFT_N=1024
       0*4,  512*4,  256*4,  768*4,  128*4,  640*4,  384*4,  896*4,   64*4,
     576*4,  320*4,  832*4,  192*4,  704*4,  448*4,  960*4,   32*4,  544*4,
     288*4,  800*4,  160*4,  672*4,  416*4,  928*4,   96*4,  608*4,  352*4,
     864*4,  224*4,  736*4,  480*4,  992*4,   16*4,  528*4,  272*4,  784*4,
     144*4,  656*4,  400*4,  912*4,   80*4,  592*4,  336*4,  848*4,  208*4,
     720*4,  464*4,  976*4,   48*4,  560*4,  304*4,  816*4,  176*4,  688*4,
     432*4,  944*4,  112*4,  624*4,  368*4,  880*4,  240*4,  752*4,  496*4,
    1008*4,    8*4,  520*4,  264*4,  776*4,  136*4,  648*4,  392*4,  904*4,
      72*4,  584*4,  328*4,  840*4,  200*4,  712*4,  456*4,  968*4,   40*4,
     552*4,  296*4,  808*4,  168*4,  680*4,  424*4,  936*4,  104*4,  616*4,
     360*4,  872*4,  232*4,  744*4,  488*4, 1000*4,   24*4,  536*4,  280*4,
     792*4,  152*4,  664*4,  408*4,  920*4,   88*4,  600*4,  344*4,  856*4,
     216*4,  728*4,  472*4,  984*4,   56*4,  568*4,  312*4,  824*4,  184*4,
     696*4,  440*4,  952*4,  120*4,  632*4,  376*4,  888*4,  248*4,  760*4,
     504*4, 1016*4,    4*4,  516*4,  260*4,  772*4,  132*4,  644*4,  388*4,
     900*4,   68*4,  580*4,  324*4,  836*4,  196*4,  708*4,  452*4,  964*4,
      36*4,  548*4,  292*4,  804*4,  164*4,  676*4,  420*4,  932*4,  100*4,
     612*4,  356*4,  868*4,  228*4,  740*4,  484*4,  996*4,   20*4,  532*4,
     276*4,  788*4,  148*4,  660*4,  404*4,  916*4,   84*4,  596*4,  340*4,
     852*4,  212*4,  724*4,  468*4,  980*4,   52*4,  564*4,  308*4,  820*4,
     180*4,  692*4,  436*4,  948*4,  116*4,  628*4,  372*4,  884*4,  244*4,
     756*4,  500*4, 1012*4,   12*4,  524*4,  268*4,  780*4,  140*4,  652*4,
     396*4,  908*4,   76*4,  588*4,  332*4,  844*4,  204*4,  716*4,  460*4,
     972*4,   44*4,  556*4,  300*4,  812*4,  172*4,  684*4,  428*4,  940*4,
     108*4,  620*4,  364*4,  876*4,  236*4,  748*4,  492*4, 1004*4,   28*4,
     540*4,  284*4,  796*4,  156*4,  668*4,  412*4,  924*4,   92*4,  604*4,
     348*4,  860*4,  220*4,  732*4,  476*4,  988*4,   60*4,  572*4,  316*4,
     828*4,  188*4,  700*4,  444*4,  956*4,  124*4,  636*4,  380*4,  892*4,
     252*4,  764*4,  508*4, 1020*4,    2*4,  514*4,  258*4,  770*4,  130*4,
     642*4,  386*4,  898*4,   66*4,  578*4,  322*4,  834*4,  194*4,  706*4,
     450*4,  962*4,   34*4,  546*4,  290*4,  802*4,  162*4,  674*4,  418*4,
     930*4,   98*4,  610*4,  354*4,  866*4,  226*4,  738*4,  482*4,  994*4,
      18*4,  530*4,  274*4,  786*4,  146*4,  658*4,  402*4,  914*4,   82*4,
     594*4,  338*4,  850*4,  210*4,  722*4,  466*4,  978*4,   50*4,  562*4,
     306*4,  818*4,  178*4,  690*4,  434*4,  946*4,  114*4,  626*4,  370*4,
     882*4,  242*4,  754*4,  498*4, 1010*4,   10*4,  522*4,  266*4,  778*4,
     138*4,  650*4,  394*4,  906*4,   74*4,  586*4,  330*4,  842*4,  202*4,
     714*4,  458*4,  970*4,   42*4,  554*4,  298*4,  810*4,  170*4,  682*4,
     426*4,  938*4,  106*4,  618*4,  362*4,  874*4,  234*4,  746*4,  490*4,
    1002*4,   26*4,  538*4,  282*4,  794*4,  154*4,  666*4,  410*4,  922*4,
      90*4,  602*4,  346*4,  858*4,  218*4,  730*4,  474*4,  986*4,   58*4,
     570*4,  314*4,  826*4,  186*4,  698*4,  442*4,  954*4,  122*4,  634*4,
     378*4,  890*4,  250*4,  762*4,  506*4, 1018*4,    6*4,  518*4,  262*4,
     774*4,  134*4,  646*4,  390*4,  902*4,   70*4,  582*4,  326*4,  838*4,
     198*4,  710*4,  454*4,  966*4,   38*4,  550*4,  294*4,  806*4,  166*4,
     678*4,  422*4,  934*4,  102*4,  614*4,  358*4,  870*4,  230*4,  742*4,
     486*4,  998*4,   22*4,  534*4,  278*4,  790*4,  150*4,  662*4,  406*4,
     918*4,   86*4,  598*4,  342*4,  854*4,  214*4,  726*4,  470*4,  982*4,
      54*4,  566*4,  310*4,  822*4,  182*4,  694*4,  438*4,  950*4,  118*4,
     630*4,  374*4,  886*4,  246*4,  758*4,  502*4, 1014*4,   14*4,  526*4,
     270*4,  782*4,  142*4,  654*4,  398*4,  910*4,   78*4,  590*4,  334*4,
     846*4,  206*4,  718*4,  462*4,  974*4,   46*4,  558*4,  302*4,  814*4,
     174*4,  686*4,  430*4,  942*4,  110*4,  622*4,  366*4,  878*4,  238*4,
     750*4,  494*4, 1006*4,   30*4,  542*4,  286*4,  798*4,  158*4,  670*4,
     414*4,  926*4,   94*4,  606*4,  350*4,  862*4,  222*4,  734*4,  478*4,
     990*4,   62*4,  574*4,  318*4,  830*4,  190*4,  702*4,  446*4,  958*4,
     126*4,  638*4,  382*4,  894*4,  254*4,  766*4,  510*4, 1022*4

Table for bit reversal for FFT_N=512
       0*4,  256*4,  128*4,  384*4,   64*4,  320*4,  192*4,  448*4,   32*4,
     288*4,  160*4,  416*4,   96*4,  352*4,  224*4,  480*4,   16*4,  272*4,
     144*4,  400*4,   80*4,  336*4,  208*4,  464*4,   48*4,  304*4,  176*4,
     432*4,  112*4,  368*4,  240*4,  496*4,    8*4,  264*4,  136*4,  392*4,
      72*4,  328*4,  200*4,  456*4,   40*4,  296*4,  168*4,  424*4,  104*4,
     360*4,  232*4,  488*4,   24*4,  280*4,  152*4,  408*4,   88*4,  344*4,
     216*4,  472*4,   56*4,  312*4,  184*4,  440*4,  120*4,  376*4,  248*4,
     504*4,    4*4,  260*4,  132*4,  388*4,   68*4,  324*4,  196*4,  452*4,
      36*4,  292*4,  164*4,  420*4,  100*4,  356*4,  228*4,  484*4,   20*4,
     276*4,  148*4,  404*4,   84*4,  340*4,  212*4,  468*4,   52*4,  308*4,
     180*4,  436*4,  116*4,  372*4,  244*4,  500*4,   12*4,  268*4,  140*4,
     396*4,   76*4,  332*4,  204*4,  460*4,   44*4,  300*4,  172*4,  428*4,
     108*4,  364*4,  236*4,  492*4,   28*4,  284*4,  156*4,  412*4,   92*4,
     348*4,  220*4,  476*4,   60*4,  316*4,  188*4,  444*4,  124*4,  380*4,
     252*4,  508*4,    2*4,  258*4,  130*4,  386*4,   66*4,  322*4,  194*4,
     450*4,   34*4,  290*4,  162*4,  418*4,   98*4,  354*4,  226*4,  482*4,
      18*4,  274*4,  146*4,  402*4,   82*4,  338*4,  210*4,  466*4,   50*4,
     306*4,  178*4,  434*4,  114*4,  370*4,  242*4,  498*4,   10*4,  266*4,
     138*4,  394*4,   74*4,  330*4,  202*4,  458*4,   42*4,  298*4,  170*4,
     426*4,  106*4,  362*4,  234*4,  490*4,   26*4,  282*4,  154*4,  410*4,
      90*4,  346*4,  218*4,  474*4,   58*4,  314*4,  186*4,  442*4,  122*4,
     378*4,  250*4,  506*4,    6*4,  262*4,  134*4,  390*4,   70*4,  326*4,
     198*4,  454*4,   38*4,  294*4,  166*4,  422*4,  102*4,  358*4,  230*4,
     486*4,   22*4,  278*4,  150*4,  406*4,   86*4,  342*4,  214*4,  470*4,
      54*4,  310*4,  182*4,  438*4,  118*4,  374*4,  246*4,  502*4,   14*4,
     270*4,  142*4,  398*4,   78*4,  334*4,  206*4,  462*4,   46*4,  302*4,
     174*4,  430*4,  110*4,  366*4,  238*4,  494*4,   30*4,  286*4,  158*4,
     414*4,   94*4,  350*4,  222*4,  478*4,   62*4,  318*4,  190*4,  446*4,
     126*4,  382*4,  254*4,  510*4

Table for bit reversal for FFT_N=256
*/
       0*4,  128*4,   64*4,  192*4,   32*4,  160*4,   96*4,  224*4,   16*4,
     144*4,   80*4,  208*4,   48*4,  176*4,  112*4,  240*4,    8*4,  136*4,
      72*4,  200*4,   40*4,  168*4,  104*4,  232*4,   24*4,  152*4,   88*4,
     216*4,   56*4,  184*4,  120*4,  248*4,    4*4,  132*4,   68*4,  196*4,
      36*4,  164*4,  100*4,  228*4,   20*4,  148*4,   84*4,  212*4,   52*4,
     180*4,  116*4,  244*4,   12*4,  140*4,   76*4,  204*4,   44*4,  172*4,
     108*4,  236*4,   28*4,  156*4,   92*4,  220*4,   60*4,  188*4,  124*4,
     252*4,    2*4,  130*4,   66*4,  194*4,   34*4,  162*4,   98*4,  226*4,
      18*4,  146*4,   82*4,  210*4,   50*4,  178*4,  114*4,  242*4,   10*4,
     138*4,   74*4,  202*4,   42*4,  170*4,  106*4,  234*4,   26*4,  154*4,
      90*4,  218*4,   58*4,  186*4,  122*4,  250*4,    6*4,  134*4,   70*4,
     198*4,   38*4,  166*4,  102*4,  230*4,   22*4,  150*4,   86*4,  214*4,
      54*4,  182*4,  118*4,  246*4,   14*4,  142*4,   78*4,  206*4,   46*4,
     174*4,  110*4,  238*4,   30*4,  158*4,   94*4,  222*4,   62*4,  190*4,
     126*4,  254*4
/*

Table for bit reversal for FFT_N=128
       0*4,   64*4,   32*4,   96*4,   16*4,   80*4,   48*4,  112*4,    8*4,
      72*4,   40*4,  104*4,   24*4,   88*4,   56*4,  120*4,    4*4,   68*4,
      36*4,  100*4,   20*4,   84*4,   52*4,  116*4,   12*4,   76*4,   44*4,
     108*4,   28*4,   92*4,   60*4,  124*4,    2*4,   66*4,   34*4,   98*4,
      18*4,   82*4,   50*4,  114*4,   10*4,   74*4,   42*4,  106*4,   26*4,
      90*4,   58*4,  122*4,    6*4,   70*4,   38*4,  102*4,   22*4,   86*4,
      54*4,  118*4,   14*4,   78*4,   46*4,  110*4,   30*4,   94*4,   62*4,
     126*4

Table for bit reversal for FFT_N=64
       0*4,   32*4,   16*4,   48*4,    8*4,   40*4,   24*4,   56*4,    4*4,
      36*4,   20*4,   52*4,   12*4,   44*4,   28*4,   60*4,    2*4,   34*4,
      18*4,   50*4,   10*4,   42*4,   26*4,   58*4,    6*4,   38*4,   22*4,
      54*4,   14*4,   46*4,   30*4,   62*4
*/
};
