/*
    This subroutine calculates the Fourier transform of an array of 256 real
    numbers.
    The array with the data points (f) is also used to store the result in.
    f(0)=F(0) and f(1)=F(128).
    The rest of the array (f(2)...f(127)) contains alternate real and imaginary
    parts for F(1) through F(127).
    F(16) is the value of the Fourier transform at the Nyquist frequency, i.e.
    at half the sample frequency.
    The power spectrum can be constructed as follows:
    - Horizontal (frequency): Sample_frequency/256*(0,1,2,...,128)
    - Vertical (power): 2/256*(f(0),sqrt(f(2)^2+f(3)^2),sqrt(f(4)^2+f(5)^2),...
                               ,sqrt(f(254)^2+f(255)^2),f(1))
*/
#define N 256
#define Nd2 128
#define Nd4 64
#define pi2 6.283185307;

void fft256(float f[])
{
    int b, i, j, k, tr_size, tr;
    float cospi2n[8]={ -1.,
                        0.,
                        7.071067812E-1,
                        9.238795325E-1,
                        9.807852804E-1,
                        9.951847267E-1,
                        9.987954562E-1,
                        9.996988187E-1};
    float sinpi2n[8]={  0.,
                        1.,
                        7.071067812E-1,
                        3.826834323E-1,
                        1.950903222E-1,
                        9.801714033E-2,
                        4.906767433E-2,
                        2.454122852E-2};
    float wb[2], temp1[2],temp2[2];
    j=0;
    for(i=1;i<Nd2;i++)
    {
        for(b=Nd4;j>=b;b=b/2)
            j=j-b;
        j=j+b;
        if(j>i)
        {
            temp1[0]=f[2*j];
            temp1[1]=f[2*j+1];
            f[2*j]=f[2*i];
            f[2*j+1]=f[2*i+1];
            f[2*i]=temp1[0];
            f[2*i+1]=temp1[1];
        }
    }
    k=0;
    for(tr_size=2;tr_size<=Nd2;tr_size=tr_size*2)
    {
        wb[0]=1.0;
        wb[1]=0.0;
        for(b=0;b<tr_size/2;b++)
        {
            for(tr=0;tr<Nd2/tr_size;tr++)
            {
                i=tr*tr_size+b;
                j=i+tr_size/2;
                temp1[0]=f[2*i];
                temp1[1]=f[2*i+1];
                temp2[0]=f[2*j];
                temp2[1]=f[2*j+1];
                f[2*i]=temp1[0]+wb[0]*temp2[0]-wb[1]*temp2[1];
                f[2*i+1]=temp1[1]+wb[0]*temp2[1]+wb[1]*temp2[0];
                f[2*j]=temp1[0]-wb[0]*temp2[0]+wb[1]*temp2[1];
                f[2*j+1]=temp1[1]-wb[0]*temp2[1]-wb[1]*temp2[0];
            }
            temp1[0]=wb[0];
            temp1[1]=wb[1];
            wb[0]=cospi2n[k]*temp1[0]-sinpi2n[k]*temp1[1];
            wb[1]=cospi2n[k]*temp1[1]+sinpi2n[k]*temp1[0];
        }
        k++;
    }
    wb[0]=1.0;
    wb[1]=0.0;
    for(b=1;b<Nd4;b++)
    {
        temp1[0]=wb[0];
        temp1[1]=wb[1];
        wb[0]=cospi2n[k]*temp1[0]-sinpi2n[k]*temp1[1];
        wb[1]=cospi2n[k]*temp1[1]+sinpi2n[k]*temp1[0];
        temp1[0]=f[2*b];
        temp1[1]=f[2*b+1];
        temp2[0]=f[2*(Nd2-b)];
        temp2[1]=f[2*(Nd2-b)+1];
        f[2*b]=0.5*(temp1[0]+temp2[0]+wb[0]*(temp1[1]+temp2[1])+wb[1]*(temp1[0]-temp2[0]));
        f[2*b+1]=0.5*(temp1[1]-temp2[1]-wb[0]*(temp1[0]-temp2[0])+wb[1]*(temp1[1]+temp2[1]));
        f[2*(Nd2-b)]=0.5*(temp1[0]+temp2[0]-wb[0]*(temp1[1]+temp2[1])-wb[1]*(temp1[0]-temp2[0]));
        f[2*(Nd2-b)+1]=0.5*(-temp1[1]+temp2[1]-wb[0]*(temp1[0]-temp2[0])+wb[1]*(temp1[1]+temp2[1]));
    }
    temp1[0]=f[0];
    temp1[1]=f[1];
    f[0]=temp1[0]+temp1[1];
    f[1]=temp1[0]-temp1[1];
}
