;*** ADCapture2fl.asm ***;
;-------------------------------------------------
; Author: Torsten Knorr, create-soft@freenet.de
; Adapted to work with float array by PepeV.
;-------------------------------------------------
;*** Register Definitions ***;
 .def R_DATA_LOW   = R0
 .def R_DATA_HIGH  = R1
 .def R_SP_LOW     = R6
 .def R_SP_HIGH    = R7
 .def R_PARAM_LOW  = R10
 .def R_PARAM_HIGH = R11
 .def R_TEMP_LOW   = R22
 .def R_TEMP_HIGH  = R23
 .def R_COUNT_LOW  = R24
 .def R_COUNT_HIGH = R25
 .def R_XP_LOW     = R26
 .def R_XP_HIGH    = R27
 .def R_ZP_LOW     = R30
 .def R_ZP_HIGH    = R31
;-------------------------------------------------
;*** ADMUX - ADC Multiplexer Selection Register ***;
 .equ ADMUX = 0x07
 .equ MUX0  = 0    ; Analog Channel and Gain Selection Bits
 .equ MUX1  = 1    ; Analog Channel and Gain Selection Bits
 .equ MUX2  = 2    ; Analog Channel and Gain Selection Bits
 .equ MUX3  = 3    ; Analog Channel and Gain Selection Bits
 .equ MUX4  = 4    ; Analog Channel and Gain Selection Bits
 .equ MUXM  = 0x1F ; Analog Channel and Gain Selection Mask
 .equ ADLAR = 5	   ; Left Adjust Result
 .equ REFS0 = 6	   ; Reference Selection Bit 0
 .equ REFS1 = 7    ; Reference Selection Bit 1
 .equ REFSM = 0xC0 ; Reference Selection Mask
;-------------------------------------------------
;*** ADCSRA - ADC Control and Status Register A ***;
 .equ ADCSRA = 0x06
 .equ ADCSR  = ADCSRA ; For compatibility
 .equ ADPS0  = 0      ; ADC Prescaler Select Bits
 .equ ADPS1  = 1      ; ADC Prescaler Select Bits
 .equ ADPS2  = 2      ; ADC Prescaler Select Bits
 .equ ADPSM  = 0x07   ; ADC Prescaler Select Mask
 .equ ADIE   = 3      ; ADC Interrupt Enable
 .equ ADIF   = 4      ; ADC Interrupt Flag
 .equ ADFR   = 5      ; ADC  Free Running Select
 .equ ADSC   = 6      ; ADC Start Conversion
 .equ ADEN   = 7      ; ADC Enable
;-------------------------------------------------
;*** ADCH - ADC Data Register High Byte ***;
 .equ ADCH  = 0x05
;-------------------------------------------------
;*** ADCL - ADC Data Register Low Byte ***;
 .equ ADCL  = 0x04
;-------------------------------------------------
;*** SREG - Status Register ***;
 .equ SREG  = 0x3F
;-------------------------------------------------
;*** MCUCR - MCU Control Register ***;
 .equ MCUCR = 0x35
 .equ IVCE  = 0    ; Interrupt Vector Change Enable
 .equ IVSEL = 1    ; Interrupt Vector Select
 .equ SM2   = 2    ; Sleep Mode Select
 .equ SM0   = 3    ; Sleep Mode Select
 .equ SM1   = 4    ; Sleep Mode Select
 .equ SE    = 5    ; Sleep Enable
 .equ SRW10 = 6    ; External SRAM Wait State Select
 .equ SRE   = 7    ; External SRAM Enable
;-------------------------------------------------
 .ifdef TagADCaptureSetVref
 ADCaptureSetVref:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_vref
    ld   R_TEMP_LOW,  Z

; mask bits REFS0/REFS1
    andi R_TEMP_LOW,  REFSM

    cbi  ADMUX,       REFS0
    cbi  ADMUX,       REFS1
    in   R_TEMP_HIGH, ADMUX
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADMUX,       R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureSetPrescaler
 ADCaptureSetPrescaler:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_prescaler
    ld   R_TEMP_LOW,  Z

; mask bits ADPS0/ADPS1/ADPS2
    andi R_TEMP_LOW,  ADPSM

    in   R_TEMP_HIGH, ADCSRA
    andi R_TEMP_HIGH, ~ADPSM
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADCSRA,      R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureSetChannel
 ADCaptureSetChannel:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_channel
    ld   R_TEMP_LOW,  Z

; mask bits MUX0/MUX1/MUX2/MUX3/MUX4
    andi R_TEMP_LOW,  MUXM

    in   R_TEMP_HIGH, ADMUX
    andi R_TEMP_HIGH, ~MUXM
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADMUX,       R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureRunFree2fl
 ADCaptureRunFree2fl:

    movw R_XP_LOW,     R_PARAM_LOW              ; move parameter stack pointer into X

    ld   R_COUNT_LOW,  X+                       ; load 2nd parameter (w_buffer_size) into R_COUNT
    ld   R_COUNT_HIGH, X+

; To keep track of the number of conversions,
; R_COUNT is decremented with each ADC until
; it reaches zero.

    ld   R_ZP_LOW,     X+                       ; load 1st parameter (p_buffer) into Z
    ld   R_ZP_HIGH,    X

    sbi  ADCSRA,       ADFR                     ; set Free Running mode
    sbi  ADCSRA,       ADEN                     ; set ADC Enable
    sbi  ADCSRA,       ADSC                     ; set Start Conversion

 ADC_WAIT:                                      ; wait for ADC to complete
    sbis ADCSRA,       ADIF
    rjmp ADC_WAIT

    sbi  ADCSRA,       ADIF                     ; clear ADC interrupt flag

    in   R_TEMP_LOW,   ADCL                     ; move ADC value in R_TEMP
    in   R_TEMP_HIGH,  ADCH

; Conversion of 12 bit ADC value (in 16 bit register) to 32 bit float.

; The 32 bits float consists of 1 sign bit + 8 bits exponent + 23 bits significant
; sign is 0
; The exponent and the significant are found by repeatedly left shifting the ADC-value until
; the highest bit with value 1 is shifted in the carry (C=1). Then:
; - the exponent is 16-n_shifts+127 = 143-n_shifts
; - the significand is the left shifted ADC value padded right with zero's to 23 bits
; The 4 bytes of the float are filled as follows (high byte to low byte):
; byte 3 = exponent>>1
; byte 2 = exponent<<7 OR HIGH(significand)>>1
; byte 1 = HIGH(significand)<<7 OR (HIGH-1)(significand)>>1
; byte 0 = 0000 0000
; The result is stored in 4 bytes of the buffer, counting from from the current value of the Z pointer

; 1. Repeat left shift ADC value until C is one.
;    The exponent is stored in R_XP_LOW. The start value is 143 (no shifts).
;    A carry from left shift R_TEMP_HIGH means the shifting is complete.
;    This is marked by setting R_XP_HIGH=1.

    ldi  R_XP_LOW,     143                      ; set starting value of the exponent to 143
    clr  R_XP_HIGH                              ; clear R_XP_HIGH

 SHIFT_LEFT_HIGH_BYTE:                          ; begin of loop of left shifting the ADC value
    lsl  R_TEMP_HIGH                            ; shift left high byte of ADC value
    brcc SHIFT_LEFT_LOW_BYTE                    ; if no carry, continue with shift left low byte
    ldi  R_XP_HIGH,    1                        ; else mark 'shifting complete'

 SHIFT_LEFT_LOW_BYTE:
    lsl  R_TEMP_LOW                             ; shift left low byte of ADC value
    brcc DEC_EXPONENT                           ; if no carry, skip next instruction
    inc  R_TEMP_HIGH                            ; add carry to high byte of data

 DEC_EXPONENT:
    dec  R_XP_LOW                               ; decrease exponent with 1
    sbrs R_XP_HIGH,    0                        ; if mark 'shifting complete', store the result
    rjmp SHIFT_LEFT_HIGH_BYTE                   ; else jump to begin of shift loop

; 2. Store the bytes of the calculated float

;   Calculate byte 3
    mov  R_XP_HIGH,    R_XP_LOW                 ; copy exponent from R_XP_LOW to R_XP_HIGH
    lsr  R_XP_HIGH                              ; byte 3 is exponent>>1
    std  Z+3,          R_XP_HIGH                ; store byte 3

;   Calculate byte 2
    mov  R_XP_HIGH,    R_XP_LOW                 ; copy exponent from R_XP_LOW to R_XP_HIGH
    ldi  R_XP_LOW,     128                      ; 2^7
    mul  R_XP_LOW,     R_XP_HIGH                ; exponent<<7 -> R_DATA_LOW
    mov  R_XP_LOW,     R_TEMP_HIGH              ; copy HIGH(significand) to R_XP_LOW
    lsr  R_XP_LOW                               ; HIGH(significand)>>1
    or   R_XP_LOW,     R_DATA_LOW               ; byte 2 is exponent<<7 OR HIGH(significand)>>1
    std  Z+2,          R_XP_LOW                 ; store byte 2

;   Calculate byte 1
    mov  R_XP_HIGH,    R_TEMP_HIGH              ; copy HIGH(significand) to R_XP_HIGH
    ldi  R_XP_LOW,     128                      ; 2^7
    mul  R_XP_LOW,     R_XP_HIGH                ; HIGH(significand)<<7 -> R_DATA_LOW
    mov  R_XP_LOW,     R_TEMP_LOW               ; copy (HIGH-1)(significand) to R_XP_LOW
    lsr  R_XP_LOW                               ; (HIGH-1)(significand)>>1
    or   R_XP_LOW,     R_DATA_LOW               ; byte 1 is HIGH(significand)<<7 OR (HIGH-1)(significand)>>1
    std  Z+1,          R_XP_LOW                 ; store byte 1

;   Calculate byte 0
    clr  R_XP_LOW                               ; byte 0 is 0
    st   Z,            R_XP_LOW                 ; store byte 0

; make Z point to the next element of the buffer (+4 bytes)
    adiw R_ZP_LOW,    4

; decrease counter
    sbiw R_COUNT_LOW, 1

; repeat ADC loop if R_COUNT is not zero
    brne ADC_WAIT

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureRelease
 ADCaptureRelease:

    cbi ADCSRA, ADEN

    ret
 .endif
;-------------------------------------------------